/* Copyright (c) 2001-2009, David A. Clunie DBA Pixelmed Publishing. All rights reserved. */

package com.pixelmed.display;

import java.awt.Color;
import java.awt.Container;
import java.awt.Dimension;
//import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.GridLayout;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;

import javax.swing.BoxLayout;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

import com.pixelmed.dicom.Attribute;
import com.pixelmed.dicom.AttributeList;
import com.pixelmed.dicom.DicomException;
import com.pixelmed.dicom.DicomInputStream;
import com.pixelmed.dicom.TagFromName;
import com.pixelmed.dicom.SOPClass;

import com.pixelmed.event.EventContext;

//import com.pixelmed.geometry.GeometryOfVolume;

/**
 * <p>This class implements a panel of icons of DICOM images inside a parent JScrollPane.</p>
 * 
 * <p>Maintains icons in a pre-defined sorted order based on DICOM attributes as they are added and removed.</p>
 * 
 * @author	dclunie
 */
public class IconListBrowser {

	protected static final int DEFAULT_ICON_SIZE = 128;
	
	protected int iconSize;

	protected JScrollPane parentScrollPane;

	protected JPanel panelOfIcons;

	
	/**
	 * <p>Set the parent scoll pane.</p>
	 *
	 * <p>Used from within constructors.</p>
	 *
	 * @param		parentScrollPane
	 */
	protected void setParentScrollPane(JScrollPane parentScrollPane) {
		this.parentScrollPane = parentScrollPane;
		parentScrollPane.setViewportView(panelOfIcons);
	}
	
	/**
	 * <p>Add a set of DICOM image files.</p>
	 *
	 * @param		dicomFileNames			a list of DICOM files
	 * @exception	DicomException			thrown if the icons cannot be extracted
	 * @exception	FileNotFoundException	thrown if a file cannot be found
	 * @exception	IOException				thrown if a file cannot be read
	 */
	public void addDicomFiles(String[] dicomFileNames) throws DicomException, FileNotFoundException, IOException {
		for (int i=0; i<dicomFileNames.length; ++i) {
			String dicomFileName = dicomFileNames[i];
			add(dicomFileName);
		}
	}

	/**
	 * <p>Build and display an (initally empty) graphical user interface view of a set of DICOM images.</p>
	 *
	 * @param		iconSize			the width and height in pixels of the icons to be created
	 * @exception	DicomException		thrown if the icons cannot be extracted
	 */
	public IconListBrowser(int iconSize) throws DicomException {
		this.iconSize = iconSize;
		
		panelOfIcons = new JPanel();
		BoxLayout panelOfIconsLayout = new BoxLayout(panelOfIcons,BoxLayout.Y_AXIS);
		//FlowLayout panelOfIconsLayout = new FlowLayout();
		//GridLayout panelOfIconsLayout = new GridLayout();
		//GridLayout panelOfIconsLayout = new GridLayout();
		//panelOfIconsLayout.setColumns(1);
		panelOfIcons.setLayout(panelOfIconsLayout);
		
		parentScrollPane = null;
	}

	/**
	 * <p>Build and display an (initally empty) graphical user interface view of a set of DICOM images.</p>
	 *
	 * @param		parentScrollPane	the scrolling pane in which the icons will be rendered
	 * @param		iconSize			the width and height in pixels of the icons to be created
	 * @exception	DicomException		thrown if the icons cannot be extracted
	 */
	public IconListBrowser(JScrollPane parentScrollPane,int iconSize) throws DicomException {
		this(iconSize);
		setParentScrollPane(parentScrollPane);
	}

	/**
	 * <p>Build and display an (initally empty) graphical user interface view of a set of DICOM images.</p>
	 *
	 * <p>Uses default icon size.</p>
	 *
	 * @param		parentScrollPane	the scrolling pane in which the icons will be rendered
	 * @exception	DicomException		thrown if the icons cannot be extracted
	 */
	public IconListBrowser(JScrollPane parentScrollPane) throws DicomException {
		this(parentScrollPane,DEFAULT_ICON_SIZE);
	}

	/**
	 * <p>Build and display a graphical user interface view of a set of DICOM image files.</p>
	 *
	 * <p>Uses default icon size.</p>
	 *
	 * @param		parentScrollPane		the scrolling pane in which the icons will be rendered
	 * @param		dicomFileNames			a list of DICOM files
	 * @exception	DicomException			thrown if the icons cannot be extracted
	 * @exception	FileNotFoundException	thrown if a file cannot be found
	 * @exception	IOException				thrown if a file cannot be read
	 */
	public IconListBrowser(JScrollPane parentScrollPane,String[] dicomFileNames) throws DicomException, FileNotFoundException, IOException {
		this(parentScrollPane);
		addDicomFiles(dicomFileNames);
	}

	/**
	 * <p>Build and display a graphical user interface view of a set of DICOM image files.</p>
	 *
	 * @param		content					a container to which will be added will be added a scrolling pane containing the icon browser
	 * @param		dicomFileNames			a list of DICOM files
	 * @exception	DicomException			thrown if the icons cannot be extracted
	 * @exception	FileNotFoundException	thrown if a file cannot be found
	 * @exception	IOException				thrown if a file cannot be read
	 */
	public IconListBrowser(Container content,String[] dicomFileNames) throws DicomException, FileNotFoundException, IOException {
		this(DEFAULT_ICON_SIZE);
		JScrollPane scrollPane = new JScrollPane();
		content.add(scrollPane);
		setParentScrollPane(scrollPane);
		addDicomFiles(dicomFileNames);
	}

	/**
	 * <p>Build and display a graphical user interface view of a set of DICOM image files.</p>
	 *
	 * @param		frame					a frame to whose content pane will be added a scrolling pane containing the icon browser
	 * @param		dicomFileNames			a list of DICOM files
	 * @exception	DicomException			thrown if the icons cannot be extracted
	 * @exception	FileNotFoundException	thrown if a file cannot be found
	 * @exception	IOException				thrown if a file cannot be read
	 */
	public IconListBrowser(JFrame frame,String[] dicomFileNames) throws DicomException, FileNotFoundException, IOException {
		this(frame.getContentPane(),dicomFileNames);
	}

	/**
	 * <p>Add an annotated icon of a DICOM image.</p>
	 *
	 * @param		list			the AttributeList containing the DICOM image
	 * @exception	DicomException	thrown if the icons cannot be extracted
	 */
	public void add(AttributeList list) throws DicomException {
		SourceImage sImg = null;
		String sopClassUID = Attribute.getSingleStringValueOrEmptyString(list,TagFromName.SOPClassUID);
		if (SOPClass.isImageStorage(sopClassUID)) {
			sImg = new SourceImage(list);
		}
		
		if (sImg != null && sImg.getNumberOfBufferedImages() > 0) {
			JPanel singleIconPanel = new JPanel();
			singleIconPanel.setLayout(new GridLayout(1,1));
			singleIconPanel.setBackground(Color.black);
			singleIconPanel.setPreferredSize(new Dimension(iconSize,iconSize));
			EventContext contextForSingleImagePanelEvents = new EventContext("ICON_" + sopClassUID);
			//GeometryOfVolume imageGeometry = getNewGeometryOfVolume(list);
			SingleImagePanel ip = new SingleImagePanel(sImg,contextForSingleImagePanelEvents/*,imageGeometry*/);
			//ip.setOrientationAnnotations(new OrientationAnnotations(list,imageGeometry),"SansSerif",Font.PLAIN,8,Color.pink);
			ip.setDemographicAndTechniqueAnnotations(new IconDemographicAndTechniqueAnnotations(list),"SansSerif",Font.PLAIN,10,Color.pink);
			singleIconPanel.add(ip);
			panelOfIcons.add(singleIconPanel);
		}
	}

	/**
	 * <p>Add an annotated icon of a DICOM image.</p>
	 *
	 * @param		dicomFileName			the name of the file containing the DICOM image
	 * @exception	DicomException			thrown if the icons cannot be extracted
	 * @exception	FileNotFoundException	thrown if a file cannot be found
	 * @exception	IOException				thrown if a file cannot be read
	 */
	public void add(String dicomFileName) throws DicomException, FileNotFoundException, IOException {
		DicomInputStream i = new DicomInputStream(new BufferedInputStream(new FileInputStream(dicomFileName)));
		AttributeList list = new AttributeList();
		list.read(i);
		add(list);
		i.close();
	}
	
	/**
	 * <p>Method for testing.</p>
	 *
	 * @param	arg	a list of DICOM image files from which to extract one icon each and display
	 */
	public static void main(String arg[]) {
		try {
			ApplicationFrame af = new ApplicationFrame("IconListBrowser");
			new IconListBrowser(af,arg);
			af.pack();
			af.setVisible(true);
		}
		catch (Exception e) {
			e.printStackTrace(System.err);
			System.exit(0);
		}
	}
}
